namespace JimmyLinqTests;

using JimmyLinq;

[TestClass]
public class ComicAnalyzerTests
{
    [TestMethod]
    public void ComicAnalyzerShouldGroupComics()
    {
        // Arrange: przygotowanie komiksów i cen.
        IEnumerable<Comic> testComics = [
             new Comic("Numer 1", 1),
             new Comic("Numer 2", 2),
             new Comic("Numer 3", 3),
        ];

        Dictionary<int, decimal> prices = new()
        {
            { 1, 20M },
            { 2, 10M },
            { 3, 1000M },
        };

        // Act: wywołanie metody grupującej komiksy według cen.
        var groups = ComicAnalyzer.GroupComicsByPrice(testComics, prices);

        // Assert: sprawdzanie grup i upewnianie się, że są poprawne.
        Assert.AreEqual(2, groups.Count());
        Assert.AreEqual(PriceRange.Tanie, groups.First().Key);
        Assert.AreEqual(2, groups.First().First().Issue);
        Assert.AreEqual("Numer 2", groups.First().First().Name);
    }

    [TestMethod]
    public void ComicAnalyzerShouldGenerateAListOfReviews()
    {
        // Arrange: przygotowywanie komiksów i recenzji.
        IEnumerable<Comic> testComics = [
            new("Numer 1", 1),
            new("Numer 2", 2),
            new("Numer 3", 3),
        ];

        IEnumerable<Review> testReviews = [
            new(1, Critics.MuddyCritic,14.5),
            new(1, Critics.RottenTornadoes,59.93),
            new(2, Critics.MuddyCritic,40.3),
            new(2, Critics.RottenTornadoes,95.11),
        ];

        // Act: wywołanie metody pobierającej recenzje.
        var actualResults =
                 ComicAnalyzer
                   .GetReviews(testComics, testReviews)
                   .ToList();

        // Assert: sprawdzanie wyników, aby ustalić, czy metoda pobrała odpowiednie recenzje.
        var expectedResults = new[]
        {
        "MuddyCritic ocenił nr 1 'Numer 1' na 14,50",
        "RottenTornadoes ocenił nr 1 'Numer 1' na 59,93",
        "MuddyCritic ocenił nr 2 'numer 2' na 40,30",
        "RottenTornadoes ocenił nr 2 'numer 2' na 95,11",
        };
        CollectionAssert.AreEqual(expectedResults, actualResults);
    }

    [TestMethod]
    public void ComicAnalyzerShouldHandleWeirdReviewScores()
    {
        // Arrange: przygotowywanie danych testowych z dziwnymi przypadkami brzegowymi.
        IEnumerable<Comic> testComics = [
            new("Numer 1", 1),
            new("Numer 2", 2),
            new("Numer 3", 3),
        ];

        IEnumerable<Review> testReviews = [
            new(1, Critics.MuddyCritic, -12.1212),
            new(1, Critics.RottenTornadoes, 391691234.48931),
            new(2, Critics.RottenTornadoes, 0),
            new(2, Critics.MuddyCritic, 40.3),
            new(2, Critics.MuddyCritic, 40.3),
            new(2, Critics.MuddyCritic, 40.3),
            new(2, Critics.MuddyCritic, 40.3),
        ];

        // Act: wywołanie metody pobierającej recenzje.
        var actualResults = ComicAnalyzer.GetReviews(testComics, testReviews).ToList();

        // Assert: upewnij się, że kod poprawnie obsługuje nietypowe przypadki brzegowe.
        List<string> expectedResults = [
                "MuddyCritic ocenił nr 1 'Numer 1' na -12,12.",
    "RottenTornadoes ocenił nr 1 'Numer 1' na 391691234,49.",
    "RottenTornadoes ocenił numer 2 'Numer 2' na 0,00.",
    "MuddyCritic ocenił nr 2 'Numer 2' na 40,30.",
    "MuddyCritic ocenił nr 2 'Numer 2' na 40,30.",
    "MuddyCritic ocenił nr 2 'Numer 2' na 40,30.",
    "MuddyCritic ocenił nr 2 'Numer 2' na 40,30.",
        ];
        CollectionAssert.AreEqual(expectedResults, actualResults);
    }


}
